unit WinImage;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls;

const
  INVALID_HANDLE_VALUE = 0;

type
  TWinImage = class(TCustomControl)
  private
    { Private declarations }
    FBackColor: TColor;
    FBorder: Boolean;
    FFillColor: TColor;
    FFont: TFont;
    FDragging: Boolean;
    FPenColor: TColor;
    FPenStyle: TPenStyle;
    FPenWidth: Integer;
    FPicture: TBitMap;
    FStretch: Boolean;
    FText: String;
    FFillStyle: TBrushStyle;
    BackgroundBitMap: TBitMap;
    PersistentBitMap: TBitMap;
    NoErase: Boolean;
    NoPaint: Boolean;
    DesignedHeight: Integer;
    DesignedWidth: Integer;
    procedure Paint; override;
    procedure SetBorder(Value: Boolean);
    procedure SetBackColor(Value: TColor);
    procedure SetDummyExtended(Value: Extended);
    procedure SetFillColor(Value: TColor);
    procedure SetDragging(Value: Boolean);
    procedure SetFillStyle(Value: TBrushStyle);
    procedure SetFont(Value: TFont);
    procedure SetPenColor(Value: TColor);
    procedure SetPenStyle(Value: TPenStyle);
    procedure SetPenWidth(Value: Integer);
    procedure SetPicture(Value: TBitMap);
    procedure SetStretch(Value: Boolean);
  protected
    { Protected declarations }
    procedure WndProc(var Message: TMessage); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y:
Integer); override;
  public
    { Public declarations }
    constructor Create(AOwner:TComponent); override;
    destructor Destroy; override;
    procedure LoadFromFile (FileName: String);
    procedure EraseDrawing;
    procedure CommitDrawing;
    procedure MoveTo(X, Y: Integer);
    procedure LineTo(X, Y: Integer);
    procedure Clear;
    procedure TextOut(X, Y: Integer; Msg: String);
    procedure Rectangle(X, Y, W, H: Integer);
    procedure Circle(X, Y, R: Integer);
  published
    { Published declarations }
    property Align;
    property BackColor: TColor read FBackColor write SetBackColor default clBtnFace;
    property Border: Boolean read FBorder write SetBorder;
    property DragCursor;
    property DragMode;
    property Dragging: Boolean read FDragging write SetDragging;
    property Enabled;
    property FillColor: TColor read FFillColor write SetFillColor default clBtnFace;
    property Font: TFont read FFont write SetFont;
    property ParentShowHint;
    property PenColor: TColor read FPenColor write SetPenColor default clBlack;
    property PenStyle: TPenStyle read FPenStyle write SetPenStyle default psSolid;
    property PenWidth: Integer read FPenWidth write SetPenWidth default 1;
    property Picture: TBitMap read FPicture write SetPicture;
    property ShowHint;
    property Stretch: Boolean read FStretch write SetStretch;
    property FillStyle: TBrushStyle read FFillStyle write SetFillStyle default bsClear;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure Register;

implementation

{ ************************************************************************* }

function MinOf(Value1, Value2: Integer): Integer;
{Return lesser of two values}
begin
if Value1 <= Value2 then
  Result := Value1
else
  Result := Value2;
end;

{ ************************************************************************* }

function MaxOf(Value1, Value2: Integer): Integer;
{Return greater of two values}
begin
if Value1 >= Value2 then
  Result := Value1
else
  Result := Value2;
end;

{ ************************************************************************* }

constructor TWinImage.Create(AOwner: TComponent);
begin
inherited Create(AOwner);
PersistentBitMap := TBitMap.Create;
BackgroundBitMap := TBitMap.Create;
FFont := TFont.Create;
FPicture := TBitMap.Create;
FBackColor := clBtnFace;
FFillColor := clBtnFace;
FFillStyle := bsClear;
FPenColor := clBlack;
FPenStyle := psSolid;
FPenWidth := 1;
with Canvas do
  begin
  Brush.Color := FFillColor;
  Brush.Style := FFillStyle;
  Pen.Color := FPenColor;
  Pen.Style := FPenStyle;
  Pen.Width := FPenWidth;
  end;
Height := 32;
Width := 32;
end;

{ ************************************************************************* }

procedure TWinImage.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y:Integer);
const
  SC_DragMove = $F012;
begin
if Dragging then
  {Allow run-time dragging}
  begin
  ReleaseCapture;
  Perform(WM_SysCommand, SC_DragMove, 0);
  end;
inherited MouseDown(Button, Shift, X, Y);
end;

{ ************************************************************************* }

procedure TWinImage.Paint;
begin
{Trash this method - we'll handle it with API calls}
end;

{ ************************************************************************* }

procedure TWinImage.WndProc(var Message: TMessage);
{Check all Windows messages to the control}
var
  X, Y: Extended;
begin
case Message.Msg of
  WM_MOVE:
    {Moving, must erase background}
    NoErase := False;
  WM_SIZE:
    {Resizing, must erase background and repaint}
    begin
    NoErase := False;
    NoPaint := False;
    end;
  WM_ERASEBKGND:
    begin
    if NoErase = True then
      begin
      {Pretend we've done it}
      Message.Result := 1;
      {Trash message}
      Exit;
      end;
    end;
  WM_PAINT:
    begin
    {Paint onto PersistentBitmap}
    NoErase := True;
    if not NoPaint then
      begin
      PersistentBitmap.Height := Height;
      PersistentBitmap.Width := Width;
      if (FStretch = True) and (not FPicture.Empty) then
        {Bitmap stretches to fit client area}
        PersistentBitmap.Canvas.StretchDraw(ClientRect, FPicture)
      else
        {Client area is size of bitmap}
        begin
        if not(FPicture.Empty) then
          begin
          PersistentBitmap.Height := FPicture.Height;
          PersistentBitmap.Width := FPicture.Width;
          PersistentBitmap.Canvas.Draw(0, 0, FPicture);
          end
        else
          begin
          PersistentBitmap.Canvas.Brush.Color := FBackColor;
          PersistentBitmap.Canvas.FillRect(ClientRect);
          end;
        end;
      with PersistentBitMap.Canvas do
        begin
        Brush.Color := FFillColor;
        Brush.Style := FFillStyle;
        Pen.Width := 1;
        Pen.Style := psSolid;
        end;
      Height := PersistentBitMap.Height;
      Width := PersistentBitMap.Width;
      if FBorder = True then
        PersistentBitmap.Canvas.Pen.Color := clBlack
      else
        PersistentBitmap.Canvas.Pen.Color := FBackColor;
      PersistentBitmap.Canvas.Rectangle(0, 0, Width, Height);
      {Save canvas background}
      BackGroundBitMap.Height := Height;
      BackGroundBitMap.Width := Width;
      BackGroundBitMap.Canvas.CopyMode := cmSrcCopy;
      BackGroundBitMap.Canvas.CopyRect(ClientRect, PersistentBitmap.Canvas, ClientRect);
      {Update the WinImage canvas}
      Canvas.CopyMode := cmSrcCopy;
      Canvas.CopyRect(ClientRect, PersistentBitmap.Canvas, ClientRect);
      NoPaint := True;
      end
    else
      {Update canvas from persistent bitmap}
      begin
      Canvas.CopyMode := cmSrcCopy;
      Canvas.CopyRect(ClientRect, PersistentBitMap.Canvas, ClientRect);
      end;
    end;
  end;
{Process rest of message}
inherited WndProc(Message);
end;

{ ************************************************************************* }

procedure TWinImage.EraseDrawing;
begin
{Restore clean background after plotting etc.}
PersistentBitMap.Canvas.CopyMode := cmSrcCopy;
PersistentBitMap.Canvas.CopyRect(ClientRect, BackGroundBitmap.Canvas, ClientRect);
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.CommitDrawing;
begin
{Save current bitmap as background}
BackgroundBitMap.Canvas.CopyMode := cmSrcCopy;
BackgroundBitMap.Canvas.CopyRect(ClientRect, PersistentBitMap.Canvas, ClientRect);
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.LoadFromFile(FileName: String);
begin
if FileExists(FileName) then
  begin
  {Load a new bitmap then repaint}
  FPicture.LoadFromFile(FileName);
  NoErase := False;
  NoPaint := False;
  Invalidate;
  end;
end;

{ ************************************************************************* }

procedure TWinImage.SetPicture(Value: TBitMap);
begin
{Assign a new bitmap}
FPicture.Assign(Value);
NoErase := False;
NoPaint := False;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.Clear;
{Remove the bitmap}
begin
FPicture.Assign(nil);
NoErase := False;
NoPaint := False;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.SetBorder(Value: Boolean);
begin
FBorder := Value;
NoErase := False;
NoPaint := False;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.SetBackColor(Value: TColor);
begin
{Floodfill control's rectangle}
FBackColor := Value;
NoErase := False;
NoPaint := False;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.SetFillColor(Value: TColor);
begin
{Fills with FillColor and Style}
FFillColor := Value;
NoErase := False;
NoPaint := False;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.SetFillStyle(Value: TBrushStyle);
begin
{Sets the fill style}
FFillStyle := Value;
NoErase := False;
NoPaint := False;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.SetFont(Value: TFont);
begin
{Sets the fill style}
FFont.Assign(Value);
end;

{ ************************************************************************* }

procedure TWinImage.SetPenColor(Value: TColor);
begin
{Sets pen color}
FPenColor := Value;
end;

{ ************************************************************************* }

procedure TWinImage.SetPenStyle(Value: TPenStyle);
begin
{Sets pen style}
FPenStyle := Value;
end;

{ ************************************************************************* }

procedure TWinImage.SetPenWidth(Value: Integer);
begin
{Sets pen width}
FPenWidth := Value;
end;

{ ************************************************************************* }

procedure TWinImage.SetDragging(Value: Boolean);
begin
{Set dragging mode}
FDragging := Value;
end;

{ ************************************************************************* }

procedure TWinImage.SetStretch(Value: Boolean);
begin
{Change the stretch property then repaint}
FStretch := Value;
NoErase := False;
NoPaint := False;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.MoveTo(X, Y: Integer);
begin
{Move to X,Y}
PersistentBitMap.Canvas.MoveTo(X, Height - Y - 1);
end;

{ ************************************************************************* }

procedure TWinImage.LineTo(X, Y: Integer);
{Draw a line to X,Y}
var
  OldX, OldY: Integer;
begin
with PersistentBitmap.Canvas do
  begin
  Pen.Color := FPenColor;
  Pen.Style := FPenStyle;
  Pen.Width := FPenWidth;
  {Save starting point}
  OldX := PenPos.X;
  OldY := PenPos.Y;
  LineTo (X, Height - Y - 1);
  {Fiddle to write last pixel}
  LineTo (OldX, OldY);
  {Restore posn to end of line}
  MoveTo(X, Height - Y - 1);
  end;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.TextOut(X, Y: Integer; Msg: String);
begin
{Write Msg at X, Y}
with PersistentBitmap do
  begin
  Canvas.Font.Assign(FFont);
  Canvas.TextOut(MinOf(X, Width), MaxOf(0, Height - Y + Canvas.Font.Height), Msg);
  end;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.Rectangle(X, Y, W, H: Integer);
begin
{Draw a filled rectangle of size (W, H) at (X, Y)}
with PersistentBitmap.Canvas do
  begin
  Pen.Color := FPenColor;
  Pen.Width := 1;
  Pen.Style := psSolid;
  Brush.Color := FPenColor;
  Brush.Style := bsSolid;
  Rectangle(X, Height - Y, X + W, Height - Y - H);
  end;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.Circle(X, Y, R: Integer);
begin
{Draw filled circle of radius (R) centred on (X, Y)}
with PersistentBitmap.Canvas do
  begin
  Pen.Color := FPenColor;
  Pen.Width := 1;
  Pen.Style := psSolid;
  Brush.Color := FPenColor;
  Brush.Style := bsSolid;
  Ellipse(X - R, Height - Y - 1 + R, X + R, Height - Y - 1 - R);
  MoveTo(X, Height - Y - 1);
  end;
Invalidate;
end;

{ ************************************************************************* }

procedure TWinImage.SetDummyExtended(Value: Extended);
begin
{Dummy write}
end;

{ ************************************************************************* }

destructor TWinImage.Destroy;
begin
FFont.Free;
FPicture.Free;
PersistentBitMap.Free;
BackGroundBitMap.Free;
inherited Destroy;
end;

{ ************************************************************************* }

procedure Register;
begin
  RegisterComponents('Samples', [TWinImage]);
end;

{ ************************************************************************* }

end.
